const axios = require('axios');
const fs = require('fs');
const path = require('path');

const ffmpegPath = require('@ffmpeg-installer/ffmpeg').path;
const ffprobePath = require('@ffprobe-installer/ffprobe').path;
const ffmpeg = require('fluent-ffmpeg');
const { generateRandomId } = require('./uniqueID');
const { getWebSettings } = require('./settings');
const sharp = require('sharp');

ffmpeg.setFfmpegPath(ffmpegPath);
ffmpeg.setFfprobePath(ffprobePath);

const { promisify } = require('util');
const { slugMaker, getVideoRenditions } = require('./dataManipulator');
const { uploadDirectoryToS3 } = require('./s3');
const { createTeaserVideo } = require('./videoProcessingFunc');
const readFile = promisify(fs.readFile);


async function verifyHlsPlaylist(playlistPath) {
  try {
    const content = await readFile(playlistPath, 'utf8');
    if (!content.includes('#EXTM3U')) throw new Error('Missing #EXTM3U header');
    if (!content.includes('#EXTINF')) throw new Error('No segments found');
    console.log(`✓ Verified HLS playlist: ${playlistPath}`);
    return true;
  } catch (err) {
    console.error(`❌ HLS verification failed:`, err.message);
    throw err;
  }
}

async function getFfmpegVersion() {
  return new Promise((resolve) => {
    ffmpeg().getAvailableFormats((err, formats) => {
      resolve(formats['hls'] ? 'With HLS support' : 'Without HLS support');
    });
  });
}

async function processVideo(slug, inputPath) {
  try {
    
    const outputDir = path.join(__basedir, "public/files/videos", slug);
    fs.mkdirSync(outputDir, { recursive: true });

    const filePath = inputPath;
    
    if (!fs.existsSync(filePath)) {
      throw new Error(`Source file not found: ${filePath}`);
    }

    //Get Video Info
    const [videoInfo, settings] = await Promise.all([
      getVideoInfo(filePath),
      getWebSettings()
    ]);
    
    // Verify FFmpeg installation
    const ffmpegVersion = await getFfmpegVersion();
    console.log(`Using FFmpeg version: ${ffmpegVersion}`);

    // Dynamic rendition selection based on source quality
    let renditions = getVideoRenditions(videoInfo.quality);

    const masterPlaylist = ['#EXTM3U'];
    const failedRenditions = [];
    
    let returnPath = '';

    for (const rendition of renditions) {
      try {
        const renditionDir = path.join(outputDir, rendition.name);
        fs.mkdirSync(renditionDir, { recursive: true });

        const outputPath = path.join(renditionDir, 'index.m3u8');
        
        // Use simpler FFmpeg command without hls_segment_filename
        await new Promise((resolve, reject) => {
          ffmpeg(filePath)
            .videoCodec('libx264')
            .audioCodec('aac')
            .videoBitrate(rendition.videoBitrate)
            .audioBitrate(rendition.audioBitrate)
            /* .size(rendition.resolution) */
            .outputOptions([
              '-f hls',
              '-hls_time 2',
              '-hls_list_size 0',
              '-start_number 0'
            ])
            .output(outputPath)
            .on('start', (cmd) => console.log(`Starting ${rendition.name}: ${cmd}`))
            .on('progress', (p) => console.log(`${rendition.name}: ${Math.round(p.percent)}%`))
            .on('end', () => {
              console.log(`✅ ${rendition.name} completed`);
              masterPlaylist.push(
                `#EXT-X-STREAM-INF:BANDWIDTH=${parseInt(rendition.videoBitrate) * 1000},RESOLUTION=${rendition.resolution}`
              );
              masterPlaylist.push(`${rendition.name}/index.m3u8`);
              returnPath = `/files/videos/${slug}/index.m3u8`;

              resolve();
            })
            .on('error', (err) => {
              console.error(`❌ ${rendition.name} failed:`, err.message);
              failedRenditions.push(rendition.name);
              reject(err);
            })
            .run();
        });

        await verifyHlsPlaylist(outputPath);

      } catch (err) {
        console.error(`Failed to process ${rendition.name}:`, err.message);
      }
    }

    if (failedRenditions.length > 0) {
      console.warn(`⚠️ Failed renditions: ${failedRenditions.join(', ')}`);
    }

    if (masterPlaylist.length > 1) {
      const masterPath = path.join(outputDir, 'index.m3u8');
      fs.writeFileSync(masterPath, masterPlaylist.join('\n'));
      console.log('✅ Master playlist created');
    } else {
      throw new Error('No renditions were successfully generated');
    }

    console.log(`Video processing completed. Output path: ${returnPath}`);

    //Generate A Teaser
    let teaser;
    try {
      teaser = await createTeaserVideo({ inputPath, outputDir, slug });
    } catch (error) {
      
    }
    

    //Upload To S3
    if(settings.website_save_to_cloud){
      const s3Prefix = `videos/${slug}`;
      console.log('⏫ Starting S3 upload...');
      await uploadDirectoryToS3(outputDir, s3Prefix);
      console.log('✅ All files uploaded to S3');
    }
    

    return { path: returnPath, slug, info: videoInfo, teaser };

  } catch (error) {
    console.error('❌ Video processing failed:', error.message);
    throw error;
  }
}


function getVideoInfo(videoPath) {
  return new Promise((resolve, reject) => {
    ffmpeg.ffprobe(videoPath, (err, metadata) => {
      if (err) return reject(err);

      const videoStream = metadata.streams.find(s => s.codec_type === 'video');
      const width = videoStream?.width || 0;
      const height = videoStream?.height || 0;
      const durationInSec = Math.floor(metadata.format.duration || 0);

      const resolution = `${width}x${height}`;

      // Quality classification
      let quality;
      if (height >= 2160) quality = '4K';
      else if (height >= 1440) quality = '2K';
      else if (height >= 1080) quality = 'Full HD';
      else if (height >= 720) quality = 'HD';
      else if (height >= 480) quality = 'SD';
      else quality = 'Low';

      resolve({
        resolution,
        quality,
        duration: durationInSec
      });
    });
  });
}

async function downloadThumbnail(thumbnail) {
  try {
    const response = await axios.get(thumbnail.url, {
      responseType: 'arraybuffer'
    });

    const buffer = Buffer.from(response.data);

    // Save buffer to a temp file
    const tempInputPath = path.join(__basedir, "public/files/images", `temp_${generateRandomId()}.webp`);
    fs.writeFileSync(tempInputPath, buffer);

    const outputPath = path.join(__basedir, "public/files/images", `${thumbnail.title}.webp`);

    let webpBuffer = await sharp(tempInputPath).rotate().resize({ fit: 'inside', withoutEnlargement: true }).toFormat('webp', { quality: 90 }).toBuffer();
                           
    fs.writeFileSync(outputPath, webpBuffer);

    // Optional: delete temp input file
    fs.unlinkSync(tempInputPath);

    return `/files/images/${thumbnail.title}.webp`;

  } catch (error) {
    console.error('Error:', error.message);
  }
}

//DOWNLOAD VIDEO FROM LINK
async function downloadVideo(video) {
  try {
    const response = await axios.get(video.videoUrl, {
      responseType: 'arraybuffer'
    });

    const buffer = Buffer.from(response.data);

    // Save buffer to a temp file
    const tempInputPath = path.join(__basedir, "public/files/videos", `temp_${generateRandomId()}.mp4`);
    fs.writeFileSync(tempInputPath, buffer);

    const result = await processVideo(slugMaker(video.title), tempInputPath);

    // Optional: delete temp input file
    fs.unlinkSync(tempInputPath);

    let thumbnail;

    if(video.thumbnail){
        thumbnail = await downloadThumbnail({ url: video.thumbnail, title: video.title })
    };

    return {
      path: result.path,
      slug: result.slug,
      teaser: result.teaser,
      info: result.info,
      thumbnail
    };

  } catch (error) {
    console.error('Error:', error.message);
  }
}
/**
 * Recursively deletes a folder and its contents
 * @param {string} folderPath
 * @param {function} callback
 */
async function deleteFolderRecursive(folderPath) {
  try {
    await fs.promises.rm(folderPath, { recursive: true, force: true });
    console.log(`✅ Deleted: ${folderPath}`);
  } catch (err) {
    console.error(`❌ Failed to delete folder: ${folderPath}`, err.message);
  }
}

module.exports = { downloadVideo, getVideoInfo, processVideo, deleteFolderRecursive }