const db = require("../models/db");
const { getRandomNumber } = require("./uniqueID");

//CREATE USER
exports.createNewUser = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_users SET ?", obj, (err, user) => {
            if (err) reject(err)
            else resolve(user)
        })
    })
};

//GET USER BY ID
exports.getUserById = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users WHERE uid = ? LIMIT 1", parseInt(userId), (err, user) => {
            if (err) reject(err)
            else resolve(user[0])
        })
    })
};


//GET USER BY EMAIL
exports.getUserByEmail = (email) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users WHERE email = ? LIMIT 1",email, (err, user) => {
            if (err) reject(err)
            else resolve(user[0])
        })
    })
}


//EDIT USER BY ID
exports.editUserById = (userId,obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_users SET ? WHERE uid = ?", [obj, parseInt(userId)], (err, user) => {
            if (err) reject(err)
            else resolve(user)
        })
    })
};

//DELETE USER BY ID
exports.deleteUserById = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_users WHERE uid = ?", parseInt(userId), (err, user) => {
            if (err) reject(err)
            else resolve(user)
        })
    })
};

// DELETE MULTIPLE USERS
exports.deleteUsers = (userIds) => {
    return new Promise((resolve, reject) => {
        // Check if userIds is an array and not empty
        if (!Array.isArray(userIds) || userIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = userIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_users WHERE uid IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};

//CREATE NEW USER TOKENS
exports.createUserTokens = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_tokens SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//DELETE USER TOKENS
exports.deleteUserTokens = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_tokens WHERE t_user_id = ?", parseInt(userId), (err, user) => {
            if (err) reject(err)
            else resolve(user)
        })
    })
};

//CREATE NEW USER RESET CODE
exports.createUserResetPasswordCode = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_reset_code SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET USER RESET CODES
exports.deleteUserCodes= (userId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_reset_code WHERE r_user_id = ?", parseInt(userId), (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET USER RESET CODE
exports.getUserResetPasswordCode = (code) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_reset_code A JOIN f_users B ON A.r_user_id = B.uid WHERE A.r_code = ?", code, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//DELETE USER RESET CODE
exports.deleteUserResetPasswordCode = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_reset_code WHERE r_user_id = ?", parseInt(userId), (err, user) => {
            if (err) reject(err)
            else resolve(user)
        })
    })
};

//GET USER BY REFERRAL CODE
exports.getUserByReferralCode = (referralCode) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users WHERE referral_code = ?", referralCode, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};


//GENERATE REFERRAL CODE
exports.generateReferralCode = async (fullname) => {
    try {
        const referralCode = fullname + getRandomNumber(5);
        const isExist = await this.getUserByReferralCode(referralCode);
        if (isExist) {
            return await this.generateReferralCode(fullname);
        } else {
            return referralCode;
        }
    } catch (error) {
        throw error.message;
    }
};

//GET USER TOKENS
exports.getUserToken = (token) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_tokens JOIN f_users ON t_user_id = uid WHERE t_token = ?", token, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//CREATE NEW USER ESSENTIALS
exports.createUserEssentials = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_users_essentials SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

// GET USER ESSENTIALS
exports.getUserEssentials = (obj) => {
    return new Promise((resolve, reject) => {
        let lockMode = ""; // Default is no lock

        if (obj.options?.allowReading !== undefined) lockMode = obj.options.allowReading ? "LOCK IN SHARE MODE" : "FOR UPDATE";

        const query = `SELECT * FROM f_users_essentials WHERE ue_user_id = ? ${lockMode}`;

        (obj.connection || db).query(query, [obj.userId], (err, data) => {
            if (err) reject(err);
            else resolve(data[0]);
        });
    });
};



//EDIT USER ESSENTIALS
exports.editUserEssentials = (obj) => {
    return new Promise((resolve, reject) => {
        (obj.connection || db).query("UPDATE f_users_essentials SET ? WHERE ue_user_id = ?", [obj.data, obj.userId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//CREATE NEW USER PREFERENCES
exports.createUserPreferences = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_users_preferences SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET USER PREFERENCES
exports.getUserPreferences = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users_preferences WHERE up_user_id = ?", obj.userId, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//EDIT USER PREFERENCES
exports.editUserPreferences = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_users_preferences SET ? WHERE up_user_id = ?", [obj.data, obj.userId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET USER PREFERENCES BY PUSH TOKEN
exports.getUserPreferencesByPushtoken = (token, connection) => {
    return new Promise((resolve, reject) => {
        (connection || db).query("SELECT * FROM f_users_preferences JOIN f_users ON up_user_id = uid WHERE pushToken = ?", token, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

// GET USERS BY MULTIPLE PUSH TOKENS (OPTIMIZED)
exports.getUsersByPushtokens = (pushTokens = [], connection) => {
    return new Promise((resolve, reject) => {
        if (!pushTokens.length) {
            resolve([]);
            return;
        }

        // Create placeholders for the tokens
        const placeholders = pushTokens.map(() => '?').join(',');
       
        const query = `SELECT * FROM f_users WHERE pushToken IN (${placeholders})`;
        
        (connection || db).query(query, pushTokens, (err, data) => {
            if (err) {
                reject(err);
            } else {
                resolve(data);
            }
        });
    });
};

// GET USERS BY ID
exports.getUsersByIds = (ids = [], connection) => {
    return new Promise((resolve, reject) => {
        if (!ids.length) {
            resolve([]);
            return;
        }

        // Create placeholders for the tokens
        const placeholders = ids.map(() => '?').join(',');
       
        const query = `SELECT * FROM f_users WHERE uid IN (${placeholders})`;
        
        (connection || db).query(query, ids, (err, data) => {
            if (err) {
                reject(err);
            } else {
                resolve(data);
            }
        });
    });
};

//GET ALL UPGRADE TYPE
exports.getAllUpgradeTypes = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_user_upgrade_types", (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//UPDATE UPGRADE TYPE
exports.updateUpgradeTypeById = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_user_upgrade_types SET ? WHERE uat_id = ?", [obj.data, obj.id], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//GET USER UPGRADE TYPE BY ID
exports.getUpgradeTypeById = (upgradeId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_user_upgrade_types WHERE uat_id = ? LIMIT 1", upgradeId, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
}

//GET USER FULL DATA
exports.getUserFullData = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE uid = ? LIMIT 1", userId, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
}

//GET USER FULL DATA BY EMAIL
exports.getUserFullDataByEmail = (email) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE email = ? LIMIT 1", email, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
}

//GET USER FULL DATA BY RECIEPIENT TYPE
exports.getUserFullDataByRecipientType = (type) => {

    let query;

    if (type == "all-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid";
    } else if (type == "regular-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE ue_user_type = 'regular'";
    } else if (type == "agent-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE ue_user_type = 'agent'";
    } else if (type == "api-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE ue_user_type = 'api'";
    }
    else {
        return Promise.reject(new Error("Invalid recipient type"));
    }

    return new Promise((resolve, reject) => {
        db.query(query, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//GET USER WITH PUSH TOKEN FULL DATA BY RECIEPIENT TYPE
exports.getUserWithTokenFullDataByRecipientType = (type) => {

    let query;

    if (type == "all-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE pushToken IS NOT NULL";
    } else if (type == "regular-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE pushToken IS NOT NULL AND ue_user_type = 'regular'";
    } else if (type == "agent-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE pushToken IS NOT NULL AND ue_user_type = 'agent'";
    } else if (type == "api-users") {
        query = "SELECT * FROM f_users JOIN f_users_essentials ON ue_user_id = uid WHERE pushToken IS NOT NULL AND ue_user_type = 'api'";
    }
    else {
        return Promise.reject(new Error("Invalid recipient type"));
    }

    return new Promise((resolve, reject) => {
        db.query(query, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}



//ADMIN GET USERS
exports.adminGetUsers = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Build dynamic filters
            let filters = `JOIN f_users_essentials ON ue_user_id = uid WHERE 1 + 1`;

            if (obj.query.type && obj.query.type !== 'all') {
                filters += ` AND ue_user_type = '${obj.query.type}'`;
            }
            if (obj.query.search) {
                filters += ` AND (fullname LIKE '%${obj.query.search}%' OR email LIKE '%${obj.query.search}%')`;
            }

            // Build SQL queries for pagination
            const query = `SELECT * FROM f_users ${filters} ORDER BY uid DESC LIMIT ${obj.query.limit} OFFSET ${obj.query.offset};`;
            
            const countQuery = `SELECT COUNT(*) as totalCount FROM f_users ${filters};`;

            const dbInstance = obj.connection || db;

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                users: dataResult,
                totalCount: countResult[0].totalCount
            });

        } catch (err) {
            return reject(err);
        }
    });
};


exports.getUserMoreinfo = (userId, connection) => {
    return new Promise((resolve, reject) => {
        const dbConn = connection || db;
        
        // Get current date for filtering
        const today = new Date().toISOString().split('T')[0];
        const firstDayOfMonth = new Date();
        firstDayOfMonth.setDate(1);
        const monthStart = firstDayOfMonth.toISOString().split('T')[0];
        
        // Run all queries concurrently
        Promise.all([
            // Today's transactions
            new Promise((resolveToday, rejectToday) => {
                dbConn.query(
                    "SELECT COALESCE(SUM(st_final_amount), 0) AS total FROM f_services_transactions WHERE st_user_id = ? AND DATE(st_created_at) = ? AND st_status = 'successful'",
                    [userId, today],
                    (err, result) => {
                        if (err) rejectToday(err);
                        else resolveToday(result[0]?.total || 0);
                    }
                );
            }),
            
            // This month's transactions
            new Promise((resolveMonth, rejectMonth) => {
                dbConn.query(
                    "SELECT COALESCE(SUM(st_final_amount), 0) AS total FROM f_services_transactions WHERE st_user_id = ? AND DATE(st_created_at) >= ? AND st_status = 'successful'",
                    [userId, monthStart],
                    (err, result) => {
                        if (err) rejectMonth(err);
                        else resolveMonth(result[0]?.total || 0);
                    }
                );
            }),
            
            // All time transactions
            new Promise((resolveAll, rejectAll) => {
                dbConn.query(
                    "SELECT COALESCE(SUM(st_final_amount), 0) AS total FROM f_services_transactions WHERE st_user_id = ? AND st_status = 'successful'",
                    [userId],
                    (err, result) => {
                        if (err) rejectAll(err);
                        else resolveAll(result[0]?.total || 0);
                    }
                );
            })
        ])
        .then(([todayTransaction, thisMonth, allTime]) => {
            resolve({
                todayTransaction: parseFloat(todayTransaction),
                thisMonth: parseFloat(thisMonth),
                allTime: parseFloat(allTime)
            });
        })
        .catch(err => {
            reject(err);
        });
    });
};