const db = require("../models/db");

//INSERT INTO SERVICES TRANSACTIONS
exports.insertIntoServicesTransactions = (obj) => {
    return new Promise(async (resolve, reject) => {
        (obj.connection || db).query("INSERT INTO f_services_transactions SET ?", obj.data, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//UPDATE SERVICES TRANSACTION
exports.updateServicesTransaction = (obj) => {
    return new Promise(async (resolve, reject) => {
        (obj.connection || db).query("UPDATE f_services_transactions SET ? WHERE st_id = ?", [obj.data, obj.transactionId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET TRANSACTION BY REFERENCE CODE
exports.getServiceTransactionByReferenceCode = (code, connection) => {
    return new Promise(async (resolve, reject) => {
        (connection || db).query("SELECT * FROM f_services_transactions WHERE st_reference_code = ?", code, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//GET TRANSACTION BY ID
exports.getServiceTransactionById = (id, connection) => {
    return new Promise(async (resolve, reject) => {
        (connection || db).query("SELECT * FROM f_services_transactions WHERE st_id = ?", id, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};


//GET SERVICES TRANSACTIONS
exports.getServicesTransactions = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Build dynamic filters
            let filters = `WHERE st_user_id = ${obj.query.userId}`;
            if (obj.query.type && obj.query.type !== 'all') {
                filters += ` AND st_type = '${obj.query.type}'`;
            }
            if (obj.query.status && obj.query.status.toLowerCase() !== 'all') {
                filters += ` AND st_status = '${obj.query.status}'`;
            }

            // Build SQL queries for pagination
            const query = `SELECT * FROM f_services_transactions ${filters} ORDER BY st_id DESC LIMIT ${obj.query.limit} OFFSET ${obj.query.offset};`;
            const countQuery = `SELECT COUNT(*) as totalCount FROM f_services_transactions ${filters};`;

            const dbInstance = obj.connection || db;

            let sums = {};

            // Only fetch totals when viewing full list from the beginning
            if (obj.query.type == "all" && obj.query.offset == 0) {

                const moneyInQuery = `
                    SELECT SUM(COALESCE(st_final_amount, st_amount)) AS depositTotal 
                    FROM f_services_transactions 
                    WHERE st_user_id = ${obj.query.userId} 
                    AND st_type = 'deposit' AND st_status = 'successful'
                    AND YEAR(st_created_at) = YEAR(CURRENT_DATE())
                    AND MONTH(st_created_at) = MONTH(CURRENT_DATE());
                `;

                const moneyOutQuery = `
                    SELECT SUM(COALESCE(st_final_amount, st_amount)) AS spentTotal 
                    FROM f_services_transactions 
                    WHERE st_user_id = ${obj.query.userId} 
                    AND st_type != 'deposit' AND st_status = 'successful'
                    AND YEAR(st_created_at) = YEAR(CURRENT_DATE())
                    AND MONTH(st_created_at) = MONTH(CURRENT_DATE());
                `;

                const [moneyInResult, moneyOutResult] = await Promise.all([
                    new Promise((res, rej) => dbInstance.query(moneyInQuery, (err, data) => err ? rej(err) : res(data))),
                    new Promise((res, rej) => dbInstance.query(moneyOutQuery, (err, data) => err ? rej(err) : res(data))),
                ]);

                sums.depositTotal = moneyInResult[0].depositTotal || 0;
                sums.spentTotal = moneyOutResult[0].spentTotal || 0;
            }

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                transactions: dataResult,
                totalCount: countResult[0].totalCount,
                sums
            });

        } catch (err) {
            return reject(err);
        }
    });
};

//GET SERVICES TRANSACTIONS
exports.adminGetServicesTransactions = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Build dynamic filters
            let filters = `LEFT JOIN f_bank_accounts ON ba_id = st_recipient JOIN f_users_essentials ON ue_user_id =st_user_id WHERE 1 + 1`;
            if (obj.query.type && obj.query.type !== 'all') {
                filters += ` AND st_type = '${obj.query.type}'`;
            }
            if (obj.query.status && obj.query.status.toLowerCase() !== 'all') {
                filters += ` AND st_status = '${obj.query.status}'`;
            }
            if (obj.query.user) {
                filters += ` AND st_user_id = '${obj.query.user}'`;
            }
            if (obj.query.search) {
                filters += ` AND st_reference_code LIKE '%${obj.query.search}%'`;
            }

            // Build SQL queries for pagination
            const query = `SELECT * FROM f_services_transactions JOIN f_users ON uid = st_user_id ${filters} ORDER BY st_id DESC LIMIT ${obj.query.limit} OFFSET ${obj.query.offset};`;
            
            const countQuery = `SELECT COUNT(*) as totalCount FROM f_services_transactions JOIN f_users ON uid = st_user_id ${filters};`;

            const dbInstance = obj.connection || db;

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                transactions: dataResult,
                totalCount: countResult[0].totalCount
            });

        } catch (err) {
            return reject(err);
        }
    });
};

//ADMIN UPDATE WITHDRAWAL HISTORY
exports.adminUpdateWithdrawalHistoryStatus = (obj, ids, connection) => {
    return new Promise(async (resolve, reject) => {
        if (!ids || !Array.isArray(ids) || ids.length === 0) {
            return resolve([]);
        }

        const conn = connection || db;
        const statusToUpdate = obj.st_status;

        // Step 1: Select ONLY pending rows with provided IDs
        const placeholders = ids.map(() => '?').join(',');
        const selectQuery = `
            SELECT * 
            FROM f_services_transactions 
            WHERE st_id IN (${placeholders}) 
            AND st_status = 'pending'
        `;

        conn.query(selectQuery, ids, (selectErr, rowsToUpdate) => {
            if (selectErr) return reject(selectErr);

            const idsToUpdate = rowsToUpdate.map(row => row.st_id);
            if (idsToUpdate.length === 0) return resolve([]); // No pending rows to update

            // Step 2: Update only pending IDs
            const updatePlaceholders = idsToUpdate.map(() => '?').join(',');
            const updateQuery = `
                UPDATE f_services_transactions 
                SET ? 
                WHERE st_id IN (${updatePlaceholders})
                AND st_status = 'pending'
            `;

            conn.query(updateQuery, [obj, ...idsToUpdate], (updateErr) => {
                if (updateErr) return reject(updateErr);

                // Step 3: Fetch updated rows with user info
                const resultQuery = `
                    SELECT t.*, u.*, e.ue_user_id, e.ue_deposit_balance 
                    FROM f_services_transactions t
                    JOIN f_users u ON u.uid = t.st_user_id
                    LEFT JOIN f_users_essentials e ON e.ue_user_id = t.st_user_id
                    WHERE t.st_id IN (${updatePlaceholders})
                `;

                conn.query(resultQuery, idsToUpdate, async (finalErr, updatedRows) => {
                    if (finalErr) return reject(finalErr);

                    // Step 4: Refund if status is 'failed'
                    if (statusToUpdate === 'failed') {
                        try {
                            await Promise.all(
                                updatedRows.map(row => {
                                    return new Promise((res, rej) => {
                                        const refundQuery = `
                                            UPDATE f_users_essentials 
                                            SET ue_deposit_balance = ue_deposit_balance + ?
                                            WHERE ue_user_id = ?
                                        `;
                                        conn.query(refundQuery, [row.st_amount, row.st_user_id], (err) => {
                                            if (err) return rej(err);
                                            res();
                                        });
                                    });
                                })
                            );
                        } catch (refundErr) {
                            return reject(refundErr);
                        }
                    }

                    // Step 5: Return updated rows
                    resolve(updatedRows);
                });
            });
        });
    });
};

exports.deleteTransactions = (transIds) => {
    return new Promise((resolve, reject) => {
        // Check if transIds is an array and not empty
        if (!Array.isArray(transIds) || transIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = transIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_services_transactions WHERE st_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};