const path = require('path');
const fs = require('fs');
const { S3Client, GetObjectCommand, ListObjectsV2Command, DeleteObjectsCommand, PutObjectCommand } = require("@aws-sdk/client-s3");
const { pipeline } = require('stream/promises');
const mime = require('mime-types');
const { getWebSettings } = require('./settings');

// Configure S3 client
const s3 = new S3Client({
  region: process.env.AWS_REGION,
  credentials: {
    accessKeyId: process.env.AWS_ACCESS_KEY_ID,
    secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
  },
});

/**
 * Deletes a folder and all its contents from S3
 * @param {string} folderPath - The folder path to delete (e.g., 'videos/my-video-slug/')
 * @returns {Promise<{deletedCount: number}>}
 */
exports.deleteS3Folder = async function(folderPath) {
  if (!folderPath.endsWith('/')) folderPath += '/';

  const listParams = { Bucket: process.env.AWS_BUCKET_NAME, Prefix: folderPath };
  const { Contents } = await s3.send(new ListObjectsV2Command(listParams));

  if (!Contents || Contents.length === 0) {
    return { deletedCount: 0 };
  }

  const deleteParams = {
    Bucket: process.env.AWS_BUCKET_NAME,
    Delete: { Objects: Contents.map(obj => ({ Key: obj.Key })) }
  };

  const { Deleted } = await s3.send(new DeleteObjectsCommand(deleteParams));
  return { deletedCount: Deleted.length };
}

/**
 * Delete a single file from S3
 * @param {string} bucketName - Your S3 bucket name
 * @param {string} fileKey - The key (path) of the file to delete
 */
exports.deleteS3File = async function(bucketName, fileKey) {
  const command = new DeleteObjectsCommand({
    Bucket: bucketName,
    Delete: {
      Objects: [
        { Key: fileKey }
      ]
    }
  });

  try {
    await s3.send(command);
    console.log(`✅ Deleted: ${fileKey}`);
  } catch (err) {
    console.error(`❌ Failed to delete ${fileKey}:`, err.message);
  }
}

/**
 * Ensures HLS stream is available locally, downloading from S3 if needed
 * @param {Object} options
 * @param {string} options.videoId - The video identifier (post.c_download_id or slug)
 * @param {string} [options.rendition] - Quality version (e.g., '360p', '720p') - optional
 * @param {string} options.localBaseDir - Base directory for local storage
 * @param {string} options.relativePath - Relative path for the streaming link
 * @returns {Promise<{downloaded: boolean, streamingLink: string, localPath: string}>}
 */
exports.ensureLocalHlsStream = async function({ videoId, rendition, localBaseDir, relativePath }) {
  // Determine paths based on whether rendition is provided
  const localDir = rendition 
    ? path.join(localBaseDir, videoId, rendition)
    : path.join(localBaseDir, videoId);
    
  const playlistPath = path.join(localDir, 'index.m3u8');
  const s3BaseKey = rendition 
    ? `videos/${videoId}/${rendition}`
    : `videos/${videoId}`;

  // Return immediately if local files exist
  if (fs.existsSync(playlistPath)) {
    const streamingLink = rendition 
      ? `${relativePath}/${videoId}/${rendition}/index.m3u8`
      : `${relativePath}/${videoId}/index.m3u8`;
    return { downloaded: false, streamingLink, localPath: playlistPath };
  }

  const { website_load_from_cloud } = await getWebSettings();
  if(website_load_from_cloud){
    try {
      // Create local directory if it doesn't exist
      if (!fs.existsSync(localDir)) {
        fs.mkdirSync(localDir, { recursive: true });
      }

      if (rendition) {
        // For renditions: download just the specified rendition files
        await downloadRenditionFiles(s3BaseKey, localDir);
      } else {
        // For streaming: download everything in the video directory
        downloadAllVideoFiles(`videos/${videoId}`, localDir);
        return { downloaded: true, streamingLink: "", localPath: "" };
      }

      const streamingLink = rendition 
        ? `${relativePath}/${videoId}/${rendition}/index.m3u8`
        : `${relativePath}/${videoId}/index.m3u8`;

      console.log(`Downloaded video ${videoId}${rendition ? ` (${rendition})` : ''} from S3 to local storage`);
      return { downloaded: true, streamingLink, localPath: playlistPath };
    } catch (error) {
      console.error(`Failed to download HLS stream for ${videoId}${rendition ? ` (${rendition})` : ''}:`, error);
      throw error;
    }
  }else return {};
  
}

/**
 * Downloads all files for a specific rendition
 */
async function downloadRenditionFiles(s3BaseKey, localDir) {
  // Download playlist file
  const s3PlaylistKey = `${s3BaseKey}/index.m3u8`;
  await downloadFromS3(s3PlaylistKey, path.join(localDir, 'index.m3u8'));
  
  // Parse playlist to find all segment files
  const playlistContent = fs.readFileSync(path.join(localDir, 'index.m3u8'), 'utf8');
  const segmentFiles = playlistContent.match(/index\d+\.ts/g) || [];
  
  // Download all segment files in parallel
  await Promise.all(segmentFiles.map(segment => {
    const segmentS3Key = `${s3BaseKey}/${segment}`;
    const segmentLocalPath = path.join(localDir, segment);
    return downloadFromS3(segmentS3Key, segmentLocalPath);
  }));
}

/**
 * Downloads all files for a video (including all renditions)
 */
async function downloadAllVideoFiles(s3VideoKey, localDir) {
  // List all files in the S3 video directory
  const { Contents } = await s3.send(new ListObjectsV2Command({
    Bucket: process.env.AWS_BUCKET_NAME,
    Prefix: s3VideoKey
  }));

  if (!Contents || Contents.length === 0) {
    throw new Error('No files found in S3 directory');
  }

  // Download all files in parallel
  await Promise.all(Contents.map(file => {
    const relativePath = file.Key.replace(s3VideoKey, '').replace(/^\//, '');
    const localPath = path.join(localDir, relativePath);
    
    // Ensure directory exists
    const fileDir = path.dirname(localPath);
    if (!fs.existsSync(fileDir)) {
      fs.mkdirSync(fileDir, { recursive: true });
    }
    
    return downloadFromS3(file.Key, localPath);
  }));
}

async function downloadFromS3(s3Key, localPath) {
  const { Body } = await s3.send(new GetObjectCommand({
    Bucket: process.env.AWS_BUCKET_NAME,
    Key: s3Key
  }));

  await pipeline(
    Body,
    fs.createWriteStream(localPath)
  );
}


/**
 * Uploads a file to S3 with proper MIME type
 */
exports.uploadFileToS3 = async function(localPath, s3Key) {
  const fileStream = fs.createReadStream(localPath);

  // Automatically detect MIME type based on file extension
  const contentType = mime.lookup(s3Key) || 'application/octet-stream';

  const params = {
    Bucket: process.env.AWS_BUCKET_NAME,
    Key: s3Key,
    Body: fileStream,
    ContentType: contentType,
  };

  try {
    await s3.send(new PutObjectCommand(params));
    console.log(`✅ Uploaded: ${s3Key} (${contentType})`);
  } catch (err) {
    console.error(`❌ Failed to upload ${s3Key}:`, err.message);
    throw err;
  }
}

/**
 * Uploads directory structure EXACTLY as-is to S3
 */
exports.uploadDirectoryToS3 = async function(localDir, s3Prefix) {
  const items = await fs.promises.readdir(localDir, { withFileTypes: true });

  // Process files and folders in order
  for (const item of items) {
    const fullPath = path.join(localDir, item.name);
    const s3Key = `${s3Prefix}/${item.name}`;

    if (item.isDirectory()) {
      // S3 doesn't need explicit folder creation - just upload files with paths
      console.log(`📁 Processing folder: ${s3Key}/`);
      await exports.uploadDirectoryToS3(fullPath, s3Key);
    } else {
      // Upload file
      await exports.uploadFileToS3(fullPath, s3Key);
    }
  }
}