const db = require("../models/db");
const { calculatePercentage } = require("./calculation");
const { ioSendSystemNotification, ioSendBalanceUpdate } = require("./events");
const { createSystemNotification } = require("./notification");
const { getReferralDetails, editReferralDetails } = require("./referral");
const { getWebSettings } = require("./settings");
const { editUserEssentials } = require("./user");

//CREATE NEW UPLINE
exports.getGenPercentageByServiceId = (serviceId) => {
  return new Promise((resolve, reject) => {
    db.query("SELECT * FROM f_referral_rewards WHERE g_service_id = ?", [serviceId], (err, data) => {
      if (err) reject(err);
      else resolve(data[0]);
    });
  });
};

//GET PERCENTAGE FOR GENERATION
exports.getGenerationsPercentage = () => {
  return new Promise((resolve, reject) => {
    db.query("SELECT * FROM f_referral_rewards", (err, data) => {
      if (err) reject(err);
      else resolve(data);
    });
  });
};

//CREATE NEW GENERATION
exports.createNewReferralReward = (obj) => {
  return new Promise((resolve, reject) => {
    db.query("INSERT INTO f_referral_rewards SET ?", obj, (err, data) => {
      if (err) reject(err);
      else resolve(data);
    });
  });
};
//EDIT GENERATION BY ID
exports.editReferralRewardById = (id, obj) => {
  return new Promise((resolve, reject) => {
    db.query(
      "UPDATE f_referral_rewards SET ? WHERE g_id = ?",
      [obj, parseInt(id)],
      (err, data) => {
        if (err) reject(err);
        else resolve(data);
      }
    );
  });
};

//EDIT GENERATION BY ID
exports.deleteReferralRewardById = (generationId) => {
  return new Promise((resolve, reject) => {
    db.query(
      "DELETE FROM f_referral_rewards WHERE g_id = ?",
      parseInt(generationId),
      (err, data) => {
        if (err) reject(err);
        else resolve(data);
      }
    );
  });
};

//CREDIT UPLINE GENERATION
exports.creditUplineGeneration = async (obj) => {
  return new Promise(async (resolve, reject) => {

    const { website_currency } = await getWebSettings();

    //Get The Referral Details For The Service
    const { downlineUserId, sharingAmount, serviceId, notificationData } = obj;
    
    if(!sharingAmount || sharingAmount < 1) return resolve();
    
    const genDetail = await this.getGenPercentageByServiceId(serviceId);
    if(!genDetail) return resolve();
    if(genDetail.g_status == "inactive") return resolve();

    //THE MAIN REFERRER DETAILS (The MR. A)
    const referrer = await getReferralDetails(downlineUserId);
    if(!referrer) return resolve();

    const amountToCredit = genDetail.g_amount_type == "fixed" ? genDetail.g_amount : calculatePercentage(genDetail.g_amount, sharingAmount);

    if(!amountToCredit) return resolve();

    //Update The User Balance
    await editUserEssentials({ 
      userId: referrer.r1_id, 
      data: { ue_referral_balance: referrer.ue_referral_balance + amountToCredit }
    });
    
    //Update Its Referral Details Balance
    await editReferralDetails(referrer.r_id, {
      r_status: 1,
      r_amount: referrer.r_amount + amountToCredit
    })
    
    //Insert Transaction To Notification
    const notificationObj = {
      sn_user_id: referrer.r1_id,
      sn_type: "referral",
      sn_title: `${website_currency}${amountToCredit.toLocaleString()} earned from Referral purchase 🚀`,
      sn_message: `Another win! A downline purchase just earned you ${website_currency}${amountToCredit.toLocaleString()}. You're on fire! 🔥`,
      sn_has_read: 0,
      sn_created_at: new Date()
    };

    //Special Notification For The Service
    if(notificationData.type == "airtime"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s airtime purchase. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "data"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s data purchase 📶. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "electricity"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s electricity purchase 💡. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "cabletv"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s TV subscription purchase 📺. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "exam-result-checker"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s exam pin purchase 🧾. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "exams-registration"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from ${notificationData.user.fullname.split(" ")[0]}'s exam registration purchase 🎓. \nKeep building your referral network! 🔥`
    
    }else if(notificationData.type == "deposit"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from a downline's deposit 💰. \nKeep building your referral network! 🔥`
    }else if(notificationData.type == "account-upgrade"){
      notificationObj.sn_message = `You have earned ${website_currency}${amountToCredit.toLocaleString()} from a downline's account upgrade 🔄. \nKeep building your referral network! 🔥`
    }

    const { insertId:systemNotificationId } = await createSystemNotification(notificationObj)
    notificationObj.sn_id = systemNotificationId;
    
    //Push Realtime System Notification To User
    ioSendSystemNotification(notificationObj);

    //Emit Balance Update Event
    ioSendBalanceUpdate({ userId: referrer.r1_id })

    return resolve();
  });
};


exports.adminGetReferralRewards = () => {
  const ignoreList = ["withdraw"]; // Add more slugs to ignore
  
  return new Promise((resolve, reject) => {
    const query = `
      SELECT s.s_id, s.s_name, s.s_slug, s.s_icon, r.g_id, r.g_amount, r.g_amount_type, r.g_status FROM f_services s LEFT JOIN f_referral_rewards r ON r.g_service_id = s.s_id WHERE s.s_slug NOT IN (?) ORDER BY s.s_id ASC`;
    
    db.query(query, [ignoreList], (err, data) => {
      if (err) {
        reject(err);
      } else {
        // Format the data for better frontend consumption
        const formattedData = data.map(item => ({
          service_id: item.s_id,
          service_name: item.s_name,
          service_slug: item.s_slug,
          service_icon: item.s_icon,
          reward: item.g_id ? {
            id: item.g_id,
            amount: item.g_amount,
            amount_type: item.g_amount_type,
            status: item.g_status
          } : null
        }));
        
        resolve(formattedData);
      }
    });
  });
};