const { stopJob, getJob } = require("../jobs/cron");
const db = require("../models/db");
const { getUserFullDataByRecipientType, getUserWithTokenFullDataByRecipientType } = require("./user");

//CREATE NEW MARKETING TEMPLATE
exports.createMarketingTemplate = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_marketing_templates SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET ALL MARKETING TEMPLATES
exports.getAllMarketingTemplates = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_templates", (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET MARKETING TEMPLATE BY ID
exports.getMarketingTemplateById = (templateId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_templates WHERE mt_id = ? LIMIT 1", [templateId], (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//UPDATE MARKETING TEMPLATE BY ID
exports.updateMarketingTemplateById = (templateId, data) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_marketing_templates SET ? WHERE mt_id = ?", [data, templateId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};


// DELETE MULTIPLE USERS MARKETING TEMPLATE
exports.deleteMarketingTemplates = (templatesId) => {
    return new Promise((resolve, reject) => {
        // Check if templatesId is an array and not empty
        if (!Array.isArray(templatesId) || templatesId.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = templatesId.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_marketing_templates WHERE mt_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};

//CREATE NEW MARKETING JOB
exports.createNewMarketingRunningJob = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_marketing_running_jobs SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET MARKETING JOB BY SCHEDULE ID
exports.getMarketingRunningJobByScheduleId = (scheduleId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_running_jobs WHERE mrj_schedule_id = ? AND  mrj_status = 'active'", [scheduleId], (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//GET ALL ACTIVE MARKETING JOB
exports.getActiveMarketingRunningJobs = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_running_jobs WHERE mrj_status = 'active'", (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//UPDATE ACTIVE MARKETING JOB
exports.updateActiveMarketingRunningJob = (jobId, data) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_marketing_running_jobs SET ? WHERE mrj_id = ?", [data, jobId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

// DELETE ACTIVE MARKETING JOB
exports.deleteMarketingRunningJob = (jobId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_marketing_running_jobs WHERE mrj_id = ?", jobId, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

// DELETE MARKETING JOB BY SCHEDULE ID
exports.deleteMarketingJobByScheduleId = (scheduleId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_marketing_running_jobs WHERE mrj_schedule_id = ?", scheduleId, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};


/* --------------------SCHEDULES---------------------  */
//CREATE NEW SCHEDULE
exports.createMarketingSchedule = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_marketing_schedules SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//UPDATE SCHEDULE
exports.updateMarketingSchedule = (id, obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_marketing_schedules SET ? WHERE ms_id = ?", [obj, id], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET ALL SCHEDULES
exports.getAllMarketingSchedules = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_schedules", (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET ALL ACTIVE SCHEDULES
exports.getAllActiveMarketingSchedules = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_schedules WHERE ms_status = 'active'", (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//GET SCHEDULE BY ID
exports.getMarketingScheduleById = (id) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_marketing_schedules WHERE ms_id = ?", id, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

// DELETE MULTIPLE USERS SCHEDULE
exports.deleteSchedules = (scheduleIds) => {
    return new Promise((resolve, reject) => {
        // Check if scheduleIds is an array and not empty
        if (!Array.isArray(scheduleIds) || scheduleIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = scheduleIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_marketing_schedules WHERE ms_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};


//GET ALL SCHEDULES
exports.handleScheduleCallback = (id) => {
    return new Promise(async (resolve, reject) => {
        try {
            
            const data = await this.getMarketingScheduleById(id);
console.log(data.ms_name, " Runned!")
            if(!data || data.ms_status != 'active'){
                stopJob(id);
                return resolve({ status: false });
            };

            const template = await this.getMarketingTemplateById(data.ms_template_id);
            
            if(!template){
                stopJob(id);
                await this.updateMarketingSchedule(id, { ms_status: "inactive" });
                return resolve({ status: false });
            };

            const isJobRunningCreated = await this.getMarketingRunningJobByScheduleId(data.ms_id);
            
            if(!isJobRunningCreated){

                //Push To Running Jobs
                const recipients = data.ms_type == "native" ? (await getUserWithTokenFullDataByRecipientType(data.ms_recipient)) : (await getUserFullDataByRecipientType(data.ms_recipient));

                await this.createNewMarketingRunningJob({
                    mrj_name: data.ms_name,
                    mrj_type: data.ms_type,
                    mrj_schedule_id: data.ms_id,
                    mrj_template_id: data.ms_template_id,
                    mrj_recipient: JSON.stringify(recipients),
                    mrj_batch_max: data.ms_batch_max,
                    mrj_batch_interval: data.ms_batch_interval
                })

                const job = getJob(id);
                
                await this.updateMarketingSchedule(id, {
                    ms_lastrun_at: new Date(),
                    ms_nextrun_at: job.nextRun()
                });

            }

            

            return resolve({ status: true });

        } catch (error) {
            return reject({ status: false, error})
        }
    })
};

