const db = require("../models/db");

//INSERT NEW LISTING
exports.insertNewGiftcardListing = (obj) => {
    return new Promise((resolve, reject) => {
      db.query("INSERT INTO f_giftcards_listing SET ?", obj, (err, data) => {
        if (err) reject(err)
        else resolve(data)
      })
    })
};

//UPDATE LISTING
exports.updateGiftcardListing = (obj) => {
    return new Promise((resolve, reject) => {
      db.query("UPDATE f_giftcards_listing SET ? WHERE gl_id = ?", [obj.data, obj.id], (err, data) => {
        if (err) reject(err)
        else resolve(data)
      })
    })
};

// DELETE MULTIPLE LISTING
exports.deleteListings = (listingIds) => {
    return new Promise((resolve, reject) => {
        // Check if listingIds is an array and not empty
        if (!Array.isArray(listingIds) || listingIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = listingIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_giftcards_listing WHERE gl_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};

//GET LISTING BY ID
exports.getGiftcardListingById = (id) => {
    return new Promise((resolve, reject) => {
      db.query("SELECT * FROM f_giftcards_listing WHERE gl_id = ?", id, (err, data) => {
        if (err) reject(err)
        else resolve(data[0])
      })
    })
};

//GET LISTING ONLY
exports.getGiftcardListingOnly = (obj) => {
    return new Promise((resolve, reject) => {
      db.query("SELECT * FROM f_giftcards_listing", (err, data) => {
        if (err) reject(err)
        else resolve(data)
      })
    })
};

//GET GIFTCARDS LISTING
exports.getGiftcardsListing = async () => {
    const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT));

    return new Promise(async (resolve, reject) => {
        try {
            // Step 1: Get all countries that have at least one giftcard
            const getCountriesWithGiftcards = () => {
                return new Promise((res, rej) => {
                    db.query(`SELECT DISTINCT c.* FROM f_countries c INNER JOIN f_giftcards g ON g.g_country_id = c.c_id`, (err, data) => {
                        if (err) rej(err);
                        else res(data);
                    });
                });
            };

            const countries = await getCountriesWithGiftcards();

            const results = await Promise.all(
                countries.map((country) =>
                    limit(() =>
                        new Promise((res, rej) => {
                            db.query(`SELECT g.*, gl.gl_name, gl.gl_icon FROM f_giftcards g INNER JOIN f_giftcards_listing gl ON gl.gl_id = g.g_list_id WHERE g.g_country_id = ?`, [country.c_id], (err, giftcards) => {
                                if (err) return rej(err);

                                res({
                                    country: country.c_name,
                                    ...country,
                                    list: giftcards
                                });
                            });
                        })
                    )
                )
            );

            resolve(results);
        } catch (error) {
            reject(error);
        }
    });
};

//GET GIFTCARDS LISTING
exports.getAllGiftcardsListing = async () => {
    return new Promise(async (resolve, reject) => {
        try {
            // Step 1: Get ALL countries (including those without giftcards)
            const getAllCountries = () => {
                return new Promise((res, rej) => {
                    db.query(`
                        SELECT 
                            c.*,
                            COUNT(g.g_id) AS has_giftcards
                        FROM f_countries c
                        LEFT JOIN f_giftcards g ON g.g_country_id = c.c_id
                        GROUP BY c.c_id
                    `, (err, countries) => {
                        if (err) rej(err);
                        else res(countries);
                    });
                });
            };

            const countries = await getAllCountries();

            // Step 2: For each country, fetch its giftcards (if any)
            const results = await Promise.all(
                countries.map((country) =>
                    new Promise((res, rej) => {
                        // Skip query if country has no giftcards
                        if (!country.has_giftcards) {
                            return res({
                                country: country.c_name,
                                ...country,
                                list: [] // Empty array for countries with no giftcards
                            });
                        }

                        // Fetch giftcards for countries that have them
                        db.query(`
                            SELECT 
                                g.*, 
                                gl.gl_name, 
                                gl.gl_icon,
                                g.g_buying_pricing,
                                g.g_selling_pricing
                            FROM f_giftcards g 
                            INNER JOIN f_giftcards_listing gl ON gl.gl_id = g.g_list_id
                            WHERE g.g_country_id = ?
                        `, [country.c_id], (err, giftcards) => {
                            if (err) return rej(err);

                            res({
                                country: country.c_name,
                                ...country,
                                list: giftcards
                            });
                        });
                    })
                )
            );

            // Step 3: Sort results (countries with giftcards first, then alphabetically)
            results.sort((a, b) => {
                // 1. Countries with giftcards come first
                if (a.list.length > 0 && b.list.length === 0) return -1;
                if (a.list.length === 0 && b.list.length > 0) return 1;
                // 2. Sort alphabetically for countries with/without giftcards
                return a.country.localeCompare(b.country);
            });

            resolve(results);
        } catch (error) {
            reject(error);
        }
    });
};

//GET GIFTCARD BY ID
exports.getGiftcardById = (giftcardId) => {
  return new Promise((resolve, reject) => {
    db.query("SELECT * FROM f_giftcards JOIN f_giftcards_listing ON gl_id = g_list_id JOIN f_countries ON c_id = g_country_id WHERE g_id = ?", giftcardId, (err, data) => {
      if (err) reject(err)
      else resolve(data[0])
    })
  })
};

//GET MOST WEEKLY PURCHASED GIFTCARDS
exports.getMostPurchasedGiftcards = (limit = 10) => {
    return new Promise((resolve, reject) => {
      db.query(`SELECT gl.gl_id, gl.gl_name, gl.gl_icon, g.g_id, g.g_name_prefix, g.g_name_suffix, MIN(g.g_buying_pricing) AS g_buying_pricing, MAX(g.g_selling_pricing) AS g_selling_pricing, COUNT(DISTINCT st.st_id) AS transaction_count, IFNULL(SUM(DISTINCT st.st_final_amount), 0) AS total_volume FROM f_giftcards_listing gl JOIN f_giftcards g ON g.g_list_id = gl.gl_id JOIN f_services_transactions st ON st.st_provider_slug = gl.gl_name AND st.st_type = 'giftcard' AND st.st_status = 'successful' AND st.st_created_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK) GROUP BY gl.gl_id, gl.gl_name, gl.gl_icon HAVING total_volume > 0 ORDER BY transaction_count DESC LIMIT ?`, [limit], (err, data) => {
        if (err) {
          console.error('Database error:', err);
          reject(err);
        } else {
          resolve(data);
        }
      });
    });
};


//INSERT NEW GIFTCARD
exports.insertNewGiftcard = (obj) => {
    return new Promise((resolve, reject) => {
      db.query("INSERT INTO f_giftcards SET ?", obj, (err, data) => {
        if (err) reject(err)
        else resolve(data)
      })
    })
};

//UPDATE GIFTCARD
exports.updateGiftcard = (obj) => {
    return new Promise((resolve, reject) => {
      db.query("UPDATE f_giftcards SET ? WHERE g_id = ?", [obj.data, obj.id], (err, data) => {
        if (err) reject(err)
        else resolve(data)
      })
    })
};

// DELETE MULTIPLE LISTING
exports.deleteGiftcards = (giftcardIds) => {
    return new Promise((resolve, reject) => {
        // Check if giftcardIds is an array and not empty
        if (!Array.isArray(giftcardIds) || giftcardIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = giftcardIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_giftcards WHERE g_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};