const moment = require('moment-timezone');
const { getUserEssentials, getUserPreferences } = require('./user');
const { getUserBankAccounts } = require('./bankAccount');
const { getUserUnreadSystemNotifications } = require('./notification');
const _ = require('lodash');
const qs = require('qs');
const { extractTime } = require('./dateTime');

exports.getDataFromObject = (mainObj, arrayOfPreferedObj) => {
    let returnObj = {};

    arrayOfPreferedObj.forEach(ar => {
        mainObj[ar] != null && (returnObj[ar] = mainObj[ar])
    });

    return returnObj
}


exports.modifyObject = function(obj, removeList) {
    if (!Array.isArray(removeList)) {
        throw new Error("removeList must be an array of property names to remove.");
    }

    const modifiedObj = { ...obj };

    for (const property of removeList) {
        if (modifiedObj.hasOwnProperty(property)) {
            delete modifiedObj[property];
        }
    }

    return modifiedObj;
}

exports.filterEmptyObject = function (obj, config = {}) {
    const filteredObj = {};

    for (const [key, value] of Object.entries(obj)) {
        if (value !== null && value !== "" && value !== undefined) {
            filteredObj[key] = value;
        } else if (value === "" && config.ignoreEmptyString) {
            filteredObj[key] = value;
        } else if (value === null && config.ignoreNullValue) {
            filteredObj[key] = value;
        }
    }

    return filteredObj;
};


//STRIP TEXT FROM HTML
exports.stripTextFromHtml = (text) => {
    const { stripHtml } = require("string-strip-html");
    return stripHtml(text).result
}


exports.copulateUserData = async (u) => {

    const user = { ...u };
    user.bankAccounts = await getUserBankAccounts(user.uid);

    //User Essentials
    const userEssentials = await getUserEssentials({ userId: user.uid }) || {};
    user.balances = {
        deposit_balance: userEssentials.ue_deposit_balance || 0,
        referral_balance: userEssentials.ue_referral_balance || 0,
    };

    user.userType = userEssentials.ue_user_type

    //User Preferences
    const userPreferences = await getUserPreferences({ userId: user.uid }) || {};
    user.preferences = this.modifyObject(userPreferences, ["up_id", "up_user_id", "up_created_at"]);

    user.transactionPin = Boolean(userEssentials?.ue_transaction_pin);

    //Get User Unread Notifications
    user.unreadNotifications = await getUserUnreadSystemNotifications(user.uid);
    
    return user
}

exports.encodeNumber = (number) => number * 100
exports.decodeNumber = (number) => number / 100


/**
 * Builds a query string from an object, removing null and empty string values.
 * @param {Object} data - The object to convert into a query string.
 * @param {Object} [options] - Optional qs.stringify options.
 * @returns {string} The resulting query string.
 */
exports.buildQueryString = function(data, options = {}) {
  const cleaned = _.pickBy(data, (value) => !_.isNil(value) && value !== '');
  return qs.stringify(cleaned, options);
}

exports.processMessageTemplateForUser = function(message, user) {
    let personalizedMessage = message
    .replace('{{fullname}}', user.fullname)
    .replace('{{email}}', user.email)
    .replace('{{currentDateTime}}', extractTime(new Date(), 'MMMM Do YYYY, h:mm:ss A'));

    return personalizedMessage;
}

exports.slugMaker = function(value) {
    return value.toLowerCase().replace(/[^\w\s-]/g, '').replace(/\s+/g, '-').replace(/--+/g, '-');
}

exports.getVideoRenditions = function(quality) {
    let renditions = [];

    if (quality === 'Low') {
      renditions = [
        { resolution: '640x360', videoBitrate: '800k', audioBitrate: '96k', name: '360p' }
      ];
    } else if (quality === 'SD') {
      renditions = [
        { resolution: '854x480', videoBitrate: '1200k', audioBitrate: '96k', name: '480p' },
        { resolution: '640x360', videoBitrate: '800k', audioBitrate: '96k', name: '360p' }
      ];
    } else if (quality === 'HD') {
      renditions = [
        { resolution: '1280x720', videoBitrate: '2800k', audioBitrate: '128k', name: '720p' },
        { resolution: '640x360', videoBitrate: '800k', audioBitrate: '96k', name: '360p' }
      ];
    } else if (quality === 'Full HD' || quality === '2K' || quality === '4K') {
      renditions = [
        { resolution: '1280x720', videoBitrate: '2800k', audioBitrate: '128k', name: '720p' },
        { resolution: '640x360', videoBitrate: '800k', audioBitrate: '96k', name: '360p' }
      ];
    } else {
      renditions = [
        { resolution: '640x360', videoBitrate: '800k', audioBitrate: '96k', name: '360p' }
      ];
    }

    return renditions;
}


exports.isURL = function (str) {
  const pattern = /^(https?:\/\/|www\.)[^\s/$.?#].[^\s]*$/i;
  return pattern.test(str);
}
exports.containsBannedWord = (text, wordsList = []) => {
  const matches = [];

  wordsList
    .filter(Boolean)
    .forEach(word => {
      const escaped = word
        .replace(/[.*+?^${}()|[\]\\]/g, '\\$&')   // escape special regex chars
        .replace(/\s+/g, '\\s+');                // support spaces as \s+

      const pattern = new RegExp(`\\b${escaped}\\b`, 'i');

      if (pattern.test(text)) {
        matches.push(word);
      }
    });

  return matches; // return array of matched banned words
};
