const { PhoneNumberUtil } = require("google-libphonenumber");
const asyncHandler = require("../../../helpers/asyncHandler");
const { getDataFromObject, copulateUserData } = require("../../../helpers/dataManipulator");
const { hashPassword, comparePassword } = require("../../../helpers/password");
const { createNewUser, getUserByEmail, deleteUserById, deleteUserTokens, createUserTokens, deleteUserResetPasswordCode, createUserResetPasswordCode, getUserResetPasswordCode, editUserById, deleteUserCodes, getUserByReferralCode, generateReferralCode, getUserById, createUserEssentials, createUserPreferences } = require("../../../helpers/user");
const { signToken, signResetPasswordToken, openToken } = require("../../../helpers/jwt");
const { getGeo } = require("../../../helpers/geo");
const { getRandomNumber } = require("../../../helpers/uniqueID");
const { userResetPasswordMailer, userRegistrationVerificationMailer, userAlertChangePasswordPinMailer, userLoginAlertMailer } = require("../../../email/mails/user");
const { getWebSettings } = require("../../../helpers/settings");
const { createNewReferral } = require("../../../helpers/referral");
const moment = require("moment");

//REFRESH LOGIN (POST)
exports.userRefreshLoginPost = asyncHandler(async (req, res, next) => {
    return res.json({ status: true, data: { user: req.user }});
})

exports.userRegisterGet = asyncHandler(async (req, res, next) => {
    let {country} = await getGeo(req.query.ip || req, req.query.ip ? "ip" : null);
    return res.json({ status: true, country })
})

exports.fetchReferrer = asyncHandler(async (req, res, next) => {
    const user = await getUserByReferralCode(req.query.ref);
    return res.json({ status: true, user })
})

//REGISTER (POST)
exports.userRegisterPost = asyncHandler(async (req, res, next) => {

    const isAdmin = req.user;

    const { website_should_verify_email } = await getWebSettings();

    //First Get Preferred Obj
    const obj = getDataFromObject(req.body, ["fullname", "email", "phone", "password", "country"]);

    if(obj?.email.includes("+")){
        const message = "Email addresses with '+' are not allowed";
        if(isAdmin) return { status: false, message  };
        else return res.json({ status: false, message })
    };

    try {

        const phoneUtil = PhoneNumberUtil.getInstance();
        const parsedNumber = phoneUtil.parse(req.body.phone);

        if (!phoneUtil.isValidNumber(parsedNumber)){
            const message = "Phone number format is incorrect";
            if(isAdmin) return { status: false, message  };
            else return res.json({ status: false, message })
        }

        const countryCode = parsedNumber.getCountryCode();
        const nationalNumber = parsedNumber.getNationalNumber().toString();

        // Replace country code with appropriate local prefix
        let localPrefix = '';
        switch (countryCode) {
            case 234: // Nigeria
                localPrefix = '0';
                break;
            // Add more cases for other countries as needed
            default:
                localPrefix = '+' + countryCode;
        }

        obj.phone2 = req.body.phone;
        obj.phone = localPrefix + nationalNumber;

    } catch (error) {
        if(isAdmin) return { status: false, message: error.message  };
        else return res.json({ status: false, message: error.message })
    }

    //Hash Password
    obj.password = await hashPassword(obj.password);

    if(!isAdmin) obj.is_verify = website_should_verify_email ? 0 : 1
    else obj.is_verify = 1;

    if (!obj.fullname.split(" ")[1]){
        const message = "Invalid Fullname"
        if(isAdmin) return { status: false, message  };
        else return res.json({ status: false, message })
    }

    //GET USERNAME FROM FULLNAME
    obj.referral_code = await generateReferralCode(obj.fullname.split(" ")[1].toUpperCase());

    //CREATE A USER
    let insertId;

    try {
        const result = await createNewUser(obj);
        insertId = result.insertId;

        await createUserEssentials({ 
            ue_user_id: insertId, 
            ue_deposit_balance: isAdmin ? req.body?.ue_deposit_balance : 0, 
            ue_referral_balance: isAdmin ? req.body?.ue_referral_balance : 0,
            ue_user_type: isAdmin ? req.body?.ue_user_type : "regular"
        });

        await createUserPreferences({ up_user_id: insertId });

        //REFERRAL
        if(req.body.ref){
            const referrer = await getUserByReferralCode(req.body.ref);
            if(referrer && referrer.uid != insertId){

                await createNewReferral({
                    r1_id: referrer.uid,
                    r2_id: insertId
                });
            
            }
            
        };

        if(!isAdmin){

            //Generate New One
            const code = getRandomNumber(5);

            const token = await signResetPasswordToken({ code, email: obj.email }, "30m");

            //Add To Database
            await createUserResetPasswordCode({
                r_user_id: insertId,
                r_code: code,
                r_token: token
            });

            obj.code = code;

            //Send Mail
            userRegistrationVerificationMailer(obj);

        }
        

    } catch (error) {
        let message;
        if (error.sqlMessage.includes("email")) {
            message = "E-mail Already in Use"
        } else if (error.sqlMessage.includes("fullname")) {
            message = "Fullname Already in Use"
        } else message = error.sqlMessage;

        //If There Is Error And User Has Been Registered
        insertId && await deleteUserById(insertId)

        //Response To Client
        if(isAdmin) return { status: false, message }
        else return res.json({ status: false, message })

    }

    //Response To User
    return isAdmin ? { status: true, data: insertId } : res.json({ status: true, message: "Registration Successful" })

})


//VERIFY REGISTER (PUT)
exports.userVerifyRegisterPut = asyncHandler(async (req, res, next) => {
    
    const code = await getUserResetPasswordCode(req.body.code);
    if (!code) return res.json({ status: false, message: "Invalid OTP Code" })

    
    //Mechanism For Checking For The Time Expiring
    try {
        await openToken(code.r_token);

        if (code.email.toLowerCase() != req.body.email?.toLowerCase()) return res.json({ status: false, message: "Invalid OTP Code" })

        await editUserById(code.r_user_id, { is_verify: 1});

        //DELETE ALL TOKENS
        await deleteUserCodes(code.r_user_id)

        let user = await getUserById(code.r_user_id);
        user = await copulateUserData(user);
        
        const token = await signToken({ id: code.r_user_id });

        //SAVED TOKEN TO DATABASE
        await createUserTokens({
            t_user_id: user.uid,
            t_token: token
        });

        
        return res.json({ status: true, message: "Account Verified", data: { token, user } });

    } catch (error) {
        if (error.message.includes("jwt expired")) return res.json({ status: false, message: "The OTP code has expired" })
        else return res.json({ status: false, message: error.message })
    }

})

//LOGIN (POST)
exports.userLoginPost = asyncHandler(async (req, res, next) => {

    //USER VARIABLE
    let user = await getUserByEmail(req.body.email || '')

    if (!user) {
        return res.json({ status: false, message: "User details not found" })
    };

    //Checking Password
    const isPasswordSame = await comparePassword(req.body.password, user.password);

    if (!isPasswordSame) {
        return res.json({ status: false, message: "Password is Incorrect" })
    }
    
    //DELETE PREVIOUS TOKENS
    await deleteUserCodes(user.uid);

    if(!user.is_verify){

        //Generate New One
        const code = getRandomNumber(5);

        const token = await signResetPasswordToken({ code, email: user.email }, "20m");

        //Add To Database
        await createUserResetPasswordCode({
            r_user_id: user.uid,
            r_code: code,
            r_token: token
        });

        user.code = code;
        user.expires = "20 minutes"
        user.headingTitle = "Verify Your Account";

        //Send Mail
        userRegistrationVerificationMailer(user);

    }
    
    //Copulate
    user = await copulateUserData(user);

    //Sign User A Token And Respond
    const token = await signToken({ id: user.uid });

    //DELETE ALL TOKENS
    await deleteUserTokens(user.uid);

    //SAVED SIGNIN TOKEN TO DATABASE
    await createUserTokens({
        t_user_id: user.uid,
        t_token: token
    });

    if(user.is_verify && user.preferences.up_email_notification){
        userLoginAlertMailer({ 
            ...user,
            headingTitle: "New Login Detected",
            loginDate: moment().format('MMM D, YYYY [at] h:mm A'), 
            loginDevice: req.body.deviceName || req.userAgent.device.toString(),
            loginIp: req.clientIp,
        })
    }
    
    if(user.is_verify) return res.json({ status: true, message: `Logged in!`, data: { token, user } });
    else return res.json({ status: true, message: `Verify your email!`, sub: "Complete your verification in just a few steps", data: { token, user } });

})

//FORGOT PASSWORD (POST)
exports.userForgotPasswordPost = asyncHandler(async (req, res, next) => {

    //USER VARIABLE
    let user = await getUserByEmail(req.body.email)

    if (!user) {
        return res.json({ status: false, message: "No User Found" })
    };

    //First Delete Previous User Reset Password Code
    await deleteUserResetPasswordCode(user.uid);

    //Generate New One
    const code = getRandomNumber(5);

    const token = await signResetPasswordToken({ code, email: user.email }, "20m");

    //Add To Database
    await createUserResetPasswordCode({
        r_user_id: user.uid,
        r_code: code,
        r_token: token
    });

    user.code = code;
    user.expires = "20 minutes";
    user.headingTitle = "Reset Your Password";

    //Send Mail
    userResetPasswordMailer(user);
    
    res.json({ status: true, message: `Mail sent` });

})

//VERIFY CODE RESET PASSWORD PASSWORD (POST)
exports.userVerifyResetPasswordPost = asyncHandler(async (req, res, next) => {

    const code = await getUserResetPasswordCode(req.body.code);
    if (!code) return res.json({ status: false, message: "Invalid OTP Code" })
    
    //Mechanism For Checking For The Time Expiring
    try {
        await openToken(code.r_token);
        if (code.email.toLowerCase() != req.body.email?.toLowerCase()) return res.json({ status: false, message: "Invalid OTP Code" })

        return res.json({ status: true, message: "OTP Code Is Valid" })
    } catch (error) {
        if (error.message.includes("jwt expired")) return res.json({ status: false, message: "The OTP code has expired" })
        else return res.json({ status: false, message: error.message })
    }

});


//RESET PASSWORD (POST)
exports.userResetPasswordPost = asyncHandler(async (req, res, next) => {

    try {

        const code = await getUserResetPasswordCode(req.body.code);
        if (!code) return res.json({ status: false, message: "Invalid OTP Code" })
        if (code.email.toLowerCase() != req.body.email?.toLowerCase()) return res.json({ status: false, message: "Invalid OTP Code" })

        //Mechanism For Checking For Token Time Expiring
        const token = await openToken(code.r_token);

        //Get user From The Token Generated
        const user = await getUserByEmail(token.email)
        if (!user) {
            return res.json({ status: false, message: "No User Found" })
        };
        
        await editUserById(user.uid, {
            password: await hashPassword(req.body.password)
        });

        //DELETE ALL TOKENS
        await deleteUserTokens(user.uid);
        
        //DELETE PREVIOUS TOKENS
        await deleteUserCodes(user.uid);

        //Alert User That Password Has Been Changed
        if(req.user.preferences.up_email_notification){
            userAlertChangePasswordPinMailer({ 
                ...user, 
                headingTitle: "Password Changed Successfully", 
                changedDate: moment().format('MMM D, YYYY [at] h:mm A'), 
                changedDevice: req.body.deviceName || req.userAgent.device.toString(),
                changedIp: req.clientIp
            });
        }
        

        res.json({ status: true, message: "Password Reset" });


    } catch (error) {
        return res.json({ status: false, message: error.message })
    }
});

//LOGOUT
exports.userLogoutOut = asyncHandler(async (req, res, next) => {
    await deleteUserTokens(req.user.uid);
    return res.json({ status: true, message: "Logged out" })
})