const asyncHandler = require("../../../helpers/asyncHandler");
const formatNumber = require("../../../helpers/formatNumber");
const { getNextOffset, getPaginationNextPrev } = require("../../../helpers/pagination");
const { getSmartLatestPost, getHotTrendingPosts, getTrendingPostsLast24Hours, getTrendingCategoryPosts,updatePostViews, updatePostReaction, getSmartLatestPostSearch } = require("../../../helpers/post");
const moment = require('moment');
const url = require('url');
const { getWebSettings } = require("../../../helpers/settings");

const ffmpegPath = require('@ffmpeg-installer/ffmpeg').path;
const path = require('path');
const fs = require('fs');
const { spawn } = require('child_process');
const { ensureLocalHlsStream } = require("../../../helpers/s3");



//HOME
exports.homepageGet = asyncHandler(async (req, res, next) => {
  const limit = parseInt(req.query.limit) || parseInt(process.env.LIMIT) || 2;
  const currentPage = parseInt(req.query.page) || 1;
  const search = req.query.search;
  const offset = getNextOffset(currentPage, limit);
  const visitedPosts = req.cookies['visited_posts'];

  if (search) {

    const production = req.query.production, quality = req.query.quality, duration = req.query.duration, sortBy = req.query.sortBy || 'latest'

    // Handle search logic only
    const [ { posts, totalCount }, webSettings ] = await Promise.all([
      getSmartLatestPostSearch({ query: { search, quality, duration, production, sortBy,limit, offset } }),
      getWebSettings()
    ]);

    const { prevPage, nextPage, totalPages } = getPaginationNextPrev({ page: currentPage, limit, totalCount });
    const currentUrl = url.parse(req.originalUrl).pathname;
    const queryParams = [];

    // Add each parameter if it exists
    if (search) queryParams.push(`search=${encodeURIComponent(search)}`);
    if (quality && quality !== 'any') queryParams.push(`quality=${encodeURIComponent(quality)}`);
    if (duration && duration !== 'any') queryParams.push(`duration=${encodeURIComponent(duration)}`);
    if (production && production !== 'any') queryParams.push(`production=${encodeURIComponent(production)}`);
    if (sortBy && sortBy !== 'any') queryParams.push(`sortBy=${encodeURIComponent(sortBy)}`);

    // Construct the pagination link
    const paginationLink = `${currentUrl}?${queryParams.join('&')}&`;

    return res.render("general/page/Blog/search", {
      title: `You searched for "${search}" - ${webSettings.website_title}`,
      thumbnail: (posts[Math.floor(Math.random() * posts.length)] || {}).c_thumbnail || '',
      description: `Search results for ${search} on ${webSettings.website_title}`,
      search,
      data: {
        posts: posts.map(post => ({ ...post, c_likes2: formatNumber(post.c_likes) })),
        pagination: { currentPage, prevPage, nextPage, totalPages, paginationLink }
      }
    });
  }

  // Default: homepage logic
  const [ mostTrending24Hours, trendingPostsData, { totalCount, posts }, webSettings ] = await Promise.all([
    getTrendingPostsLast24Hours({ query: { limit: 10, offset: 0 } }),
    getHotTrendingPosts({ query: { limit: 10, offset: 0 } }),
    getSmartLatestPost({ query: { visitedPosts, limit, offset } }),
    getWebSettings()
  ]);

  const { prevPage, nextPage, totalPages } = getPaginationNextPrev({ page: currentPage, limit, totalCount });

  const currentUrl = url.parse(req.originalUrl).pathname;
  let paginationLink = currentUrl;
  if (req.query.search) {
    paginationLink = `${paginationLink}?search=${req.query.search}&`;
  } else {
    paginationLink = `?`;
  }

  const categoriesPosts = await getTrendingCategoryPosts(JSON.parse(webSettings.website_fallback_categories), 5, 20);

  // Format the posts for rendering

  const { website_title, website_tagline } = webSettings;

  let homeTitle = "";
  if (mostTrending24Hours.posts.length) {
    homeTitle = `Trending Now: ${mostTrending24Hours.posts[0].c_title}`;
  } else if (trendingPostsData.posts.length) {
    homeTitle = `Trending: ${trendingPostsData.posts[0].c_title}`;
  } else if (posts.length) {
    homeTitle = `Latest: ${posts[0].c_title}`;
  } else {
    homeTitle = `Welcome to ${website_title}`;
  }

  return res.render("general/page/Blog", {
    title: `${homeTitle} - ${website_tagline}`,
    data: {
      mostTrending: mostTrending24Hours.posts.map(post => ({
        ...post,
        c_likes2: formatNumber(post.c_likes),
        formatDate: moment(post.c_created_at).fromNow()
      })),
      trendingPosts: trendingPostsData.posts.map(post => ({
        ...post,
        c_likes2: formatNumber(post.c_likes)
      })),
      latestPosts: {
        posts: posts.map(post => ({
          ...post,
          c_likes2: formatNumber(post.c_likes)
        })),
        pagination: {
          currentPage,
          prevPage,
          nextPage,
          totalPages,
          paginationLink
        }
      },
      categoriesPosts
    }
  });
});



//UPDATE PUT
exports.updatePostPut = asyncHandler(async (req, res, next) => {

    if(req.query.type == "view"){
        await updatePostViews(req.params.slug);
        return res.json({ status: true, message: "Post view updated" });

    }else if(req.query.type == "like" || req.query.type == "dislike"){
        await updatePostReaction(req.params.slug, req.query.type, req.query.action);
        return res.json({ status: true, message: "Post reaction updated" });

    }
    else return res.json({ status: false, message: "Invalid type" });

})

const watermarkConfig = {
  position: 'main_w-overlay_w-10:main_h-overlay_h-30',
  scale: 0.3,
  removeWatermark: null // Optional: { x: 10, y: 10, w: 100, h: 50 }
};


exports.downloadVideoMp4Get = asyncHandler(async (req, res, next) => {
  try {
    const { slug, rendition } = req.params;
    const renditionDir = path.join(__basedir, 'public/files/videos', slug, rendition);
    
    // Check if file exists locally, otherwise use S3
    const { downloaded, streamingLink } = await ensureLocalHlsStream({
      videoId: slug,
      rendition: rendition,
      localBaseDir: path.join(__basedir, 'public/files/videos'),
      relativePath: '/files/videos'
    });
    
    if (!downloaded && !fs.existsSync(path.join(__basedir, "public", streamingLink))) {
      throw new Error('Video not available');
    }

    const { website_watermark } = await getWebSettings();


    if (website_watermark) {
      const watermarkFilePath = path.join(__basedir, 'public', website_watermark);
      if (fs.existsSync(watermarkFilePath)) {
        watermarkConfig.watermarkPath = watermarkFilePath;
      }
    }

    const ffmpegArgs = ['-i', 'index.m3u8'];

    if (watermarkConfig.watermarkPath) {
      ffmpegArgs.push('-i', watermarkConfig.watermarkPath);
    }

    let filterComplex = [];
    if (watermarkConfig.watermarkPath) {
      filterComplex.push(
        `[1:v]scale=${watermarkConfig.scale}*iw:${watermarkConfig.scale}*ih[wm]`,
        `[0:v][wm]overlay=${watermarkConfig.position}`
      );
    }

    if (filterComplex.length > 0) {
      ffmpegArgs.push('-filter_complex', filterComplex.join(';'));
    }

    ffmpegArgs.push(
      '-c:v', 'libx264',
      '-preset', 'fast',
      '-c:a', 'aac',
      '-b:a', '128k',
      '-ar', '44100',
      '-movflags', 'frag_keyframe+empty_moov',
      '-bsf:a', 'aac_adtstoasc',
      '-f', 'mp4',
      'pipe:1'
    );

    const ffmpeg = spawn(ffmpegPath, ffmpegArgs, {
      cwd: renditionDir,
      stdio: ['ignore', 'pipe', 'pipe']
    });

    res.setHeader('Content-Disposition', `attachment; filename="${slug}-${rendition}.mp4"`);
    res.setHeader('Content-Type', 'video/mp4');

    ffmpeg.stdout.pipe(res);

    ffmpeg.stderr.on('data', (data) => {
      const str = data.toString();
      if (/error|warning|failed|invalid/i.test(str)) {
        console.error(`FFmpeg: ${str.trim()}`);
      }
    });

    req.on('close', () => {
      ffmpeg.kill('SIGKILL');
    });

    ffmpeg.on('close', (code) => {
      if (code !== 0) {
        console.error(`FFmpeg exited with code ${code}`);
        if (!res.headersSent) {
          res.render("general/page/404", { 
            title: 'Video conversion failed',
            description: 'An error occurred while converting the video.'
          });
        }
      }
    });

  } catch (error) {
    return res.render("general/page/404", { 
      title: 'Video downloading failed',
      description: 'An error occurred while downloading the video.'
    });
  }
});

