const asyncHandler = require("../../../helpers/asyncHandler");
const { slugMaker } = require("../../../helpers/dataManipulator");
const { getPaginationNextPrev, getNextOffset } = require("../../../helpers/pagination");
const { createNewPost, createPostCategoryPivot, editPostById, filterPostData } = require("../../../helpers/post");
const { getScrappedContents, deleteScrappedContent, editScrappedContent } = require("../../../helpers/scrapped");
const path = require("path");
const fs = require("fs");
const { downloadVideo } = require("../../../helpers/videoProcessing");
const { generateRandomId } = require("../../../helpers/uniqueID");
const { getWebSettings } = require("../../../helpers/settings");
const { uploadFileToS3 } = require("../../../helpers/s3");

//ADMIN SCRAPPED CONTENTS
exports.adminScrappedContentsGet = asyncHandler(async (req, res, next) => {

    const limit = req.query.limit || parseInt(process.env.LIMIT), currentPage = parseInt(req.query.page) || 1, search = req.query.search;

    const { totalCount, contents } = await getScrappedContents({
        query: { search, limit, offset: getNextOffset(currentPage, limit) }
    });

    const { prevPage, nextPage, totalPages } = getPaginationNextPrev({ page: currentPage, limit, totalCount});


    res.json({ status: true, data: { contents, pagination: { currentPage, prevPage, nextPage, totalPages } }})
});

//APPROVE SCRAPPED CONTENT
exports.adminScrappedContentsPost = asyncHandler(async (req, res, next) => {

    try{

        const settings = await getWebSettings();
        
        const filterResult = filterPostData({ title: req.body.sc_title, description: req.body.sc_description, tags: JSON.parse(req.body.tags || []).join(" ") }, JSON.parse(settings.website_banned_words));
        if(!filterResult.status) {
            return res.json(filterResult);
        };

        const slug = slugMaker(req.body.sc_title);

        const postData = {
            c_title: req.body.sc_title,
            c_slug: slug,
            c_production: req.body.sc_production,
            c_tags: req.body.tags,
            c_author: req.user.uid
        };

        if (settings.website_show_source_credit) {
            postData.c_site = req.body.sc_site;
        }

        const { insertId } = await createNewPost(postData);
    
        const categoryIds = JSON.parse(req.body.categories);
        await createPostCategoryPivot(insertId, categoryIds);


        res.json({ status: true, message: "Post approved"})
    
        //Do The Thumbnail
        let thumbnail;

        if(req.compressedImgs.length){

            const filename = `${slug}.webp`;
            const filePath = path.join(__basedir, 'public/files/images', filename);
            await fs.promises.writeFile(filePath, req.compressedImgs[0].buffer);
            
            await editPostById({ postId: insertId, data: { c_thumbnail: `/files/images/${filename}`} });

            thumbnail = `/files/images/${filename}`
        }

        const result = await downloadVideo({
            title: `${slug}-${generateRandomId()}`,
            thumbnail: !thumbnail ? req.body.sc_thumbnail : null,
            videoUrl: req.body.sc_download_link
        });
        
        const data = { 
            c_download_link: result.path, 
            c_download_id: result.slug, 
            c_teaser_link: result.teaser,
            c_resolution: result?.info?.resolution,
            c_quality: result?.info?.quality,
            c_duration: result?.info?.duration,
            c_status: "published"
        };

        data.c_thumbnail = result.thumbnail || thumbnail;
        
        if(data.c_thumbnail){

            if(settings.website_save_to_cloud) {
                const s3Key = `images/${result.slug}.webp`;
                uploadFileToS3(path.join(__basedir, "public", data.c_thumbnail), s3Key);
                data.c_thumbnail_backup = s3Key
            }
        };

        await editPostById({ postId: insertId, data });
        
        await editScrappedContent(req.body.sc_id, { sc_status: 'done' });

    } catch (error) {console.log(error)
        let message;

        if (error.sqlMessage?.includes("c_link")) {
            message = "Post already exists"
        } else message = error?.sqlMessage || "Something went wrong";

        //Response To Client
        return res.json({ status: false, message })

    };
      
})

//SCRAPPED CONTENT (DELETE)
exports.adminScrappedContentsDelete = asyncHandler(async (req, res, next) => {
    
    const deletedCount = await deleteScrappedContent(req.body)

    let message = "";
    if(deletedCount > 1) message = `${deletedCount} Contents Deleted`
    else if(deletedCount == 1) message = "Content Deleted";
    else message = "No Content Deleted"

    res.json({ status: Boolean(deletedCount), message });

});
