const { sendMailForMarketing } = require("../../../email/mails/user");
const asyncHandler = require("../../../helpers/asyncHandler");
const { processMessageTemplateForUser } = require("../../../helpers/dataManipulator");
const { ioSendSystemNotification } = require("../../../helpers/events");
const { getMarketingTemplateById, createNewMarketingRunningJob, createMarketingSchedule, updateMarketingSchedule, deleteSchedules, handleScheduleCallback, getMarketingScheduleById, deleteMarketingJobByScheduleId } = require("../../../helpers/marketing");
const { pushNotification, createSystemNotification } = require("../../../helpers/notification");
const { getWebSettings } = require("../../../helpers/settings");
const { getUserFullDataByEmail, getUsersByPushtokens, getUsersByIds, getUserFullDataByRecipientType } = require("../../../helpers/user");
const { scheduleMarketingJobOnCron, stopJob } = require("../../../jobs/cron");


exports.scheduleMarketingJob = asyncHandler(async (req, res, next) => {

  if(!req.body.marketingType) return res.json({ status: false, message: "Marketing Type is required" });

  let template = await getMarketingTemplateById(req.body.templateId || 0);

  if(!template) return res.json({ status: false, message: "Template not found or has been deleted"});

  if(!(req.body.recipients == "all-users" || req.body.recipients == "regular-users" || req.body.recipients == "agent-users" || req.body.recipients == "api-users")) return res.json({ status: false, message: "Invalid Recipients" });

  const { website_mail_max_per_batch, website_mail_max_per_batch_interval } = await getWebSettings();

  const data = {
    ms_name: req.body.scheduleName || template.mt_name,
    ms_type: req.body.marketingType,
    ms_template_id: template.mt_id,
    ms_recipient: req.body.recipients,
    ms_batch_max: Number(req.body.maxPerBatch) || website_mail_max_per_batch,
    ms_batch_interval: Number(req.body.maxPerBatchInterval) || website_mail_max_per_batch_interval,
    ms_cron_schedule: req.body.cronTime,
    ms_status: req.body.status || 'inactive'
  };

  const { insertId } = await createMarketingSchedule(data);
  data.ms_id = insertId;
  data.ms_created_at = new Date();

  try {
    
    if(data.ms_status == "active"){

      //Now Schedule The Job To Croner
      const job = await scheduleMarketingJobOnCron(data.ms_cron_schedule, insertId, async ()=> handleScheduleCallback(insertId));

      const nextRun = job.nextRun();

      await updateMarketingSchedule(insertId, { ms_nextrun_at: nextRun });

      data.ms_nextrun_at = nextRun;

    }
    
  } catch (error) {
    await deleteSchedules([insertId]);
    return res.json({ status: false, message: error?.message || "Something terrible went wrong" });
  }

  return res.json({ status: true, message: "Schedule Created", data })

})

//EDIT SCHEDULE
exports.scheduleMarketingPut = asyncHandler(async (req, res, next) => {

  if(!(req.body.recipients == "all-users" || req.body.recipients == "regular-users" || req.body.recipients == "agent-users" || req.body.recipients == "api-users")) return res.json({ status: false, message: "Invalid Recipients" });

  const schedule = await getMarketingScheduleById(req.body.id);
  if(!schedule) return res.json({ status: false, message: "Schedule not found or has been deleted" })

  const updatedObj = {
    ...schedule,
    ms_name: req.body.scheduleName || schedule.ms_name,
    ms_template_id: req.body.templateId || schedule.ms_template_id,
    ms_recipient: req.body.recipients || schedule.ms_recipient,
    ms_cron_schedule: req.body.cronTime || schedule.ms_cron_schedule,
    ms_batch_max: req.body.maxPerBatch || schedule.ms_batch_max,
    ms_batch_interval: req.body.maxPerBatchInterval || schedule.ms_batch_interval,
    ms_status: req.body.status || schedule.ms_status,
  };

  //Check If CronTime Is Changed
  if((schedule.ms_cron_schedule != updatedObj.ms_cron_schedule) || (schedule.ms_status != updatedObj.ms_status)){
    
    //Stop Current Job
    stopJob(req.body.id);

    let job;

    await deleteMarketingJobByScheduleId(req.body.id);

    if((schedule.ms_cron_schedule != updatedObj.ms_cron_schedule && updatedObj.ms_status == 'active') || (schedule.ms_status != updatedObj.ms_status && updatedObj.ms_status == 'active')){

      //Now Schedule New Job
      job = await scheduleMarketingJobOnCron(updatedObj.ms_cron_schedule, Number(req.body.id), async ()=> handleScheduleCallback(Number(req.body.id)));

    }
    

    updatedObj.ms_nextrun_at = job?.nextRun() || null;

  };

  await updateMarketingSchedule(req.body.id, updatedObj);

  return res.json({ status: true, message: "Schedule Updated", data: updatedObj });
  
});

//DELETE SCHEDULE
exports.scheduleMarketingDelete = asyncHandler(async (req, res, next) => {


  const [deletedCount, deletedJobs] = await Promise.all([
    deleteSchedules(req.body),
    req.body.map(a => stopJob(Number(a)))
  ])

  let message = "";
  if(deletedCount > 1) message = `${deletedCount} Schedules Deleted`
  else if(deletedCount == 1) message = "Schedule Deleted";
  else message = "No Schedule Deleted"

  res.json({ status: Boolean(deletedCount), message })

});

exports.marketingSendMail = asyncHandler(async (req, res, next) => {

  let template, rawRecipients, recipients;

  if(req.body.templateId){
    template = await getMarketingTemplateById(req.body.templateId);
    if(!template) return res.json({ status: false, message: "Template not found or has been deleted"});

    if(req.body.recipients == "all-users" || req.body.recipients == "regular-users" || req.body.recipients == "agent-users" || req.body.recipients == "api-users"){
      recipients = await getUserFullDataByRecipientType(req.body.recipients);
    }
  }


  //Incase Of Sending To Specific Emails
  if(!recipients){

    rawRecipients = [...new Set(req.body.recipients?.split(",").map(r => r.trim()).map(email => email.toLowerCase()))];

    // Parallel processing of recipients
    recipients = (await Promise.all(
      rawRecipients.map(async email => {
          try {
              const user = await getUserFullDataByEmail(email);
              return user || { email, exists: false }; // Fallback if user not found
          } catch (error) {
              return { email, error: true }; // Error handling
          }
      })
    )).filter(r => !r.error && r.exists !== false);
  }

  if(!recipients.length) return res.json({ status: false, message: "No valid recipients found" });

  let readyData, attachments;

  if(recipients.length == 1){

    [readyData, attachments] = await Promise.all([
      Promise.resolve(recipients.map(r => ({ ...r, subject: processMessageTemplateForUser(req.body.subject, r), message: processMessageTemplateForUser(req.body.message, r) }))),
      
      Promise.resolve(req.files.map(file => ({
          filename: file.originalname,
          content: file.buffer,
          contentType: file.mimetype
        }))
      )
    ]);

  }

    
  if(recipients.length > 1){

    const { website_mail_max_per_batch, website_mail_max_per_batch_interval } = await getWebSettings();

    //Push To Running Jobs
    await createNewMarketingRunningJob({
      mrj_name: template.mt_name,
      mrj_type: "email",
      mrj_template_id: template.mt_id,
      mrj_recipient: JSON.stringify(recipients),
      mrj_batch_max: req.body.maxPerBatch || website_mail_max_per_batch,
      mrj_batch_interval: req.body.maxPerBatchInterval || website_mail_max_per_batch_interval,
    })
    
    return res.json({ status: true, message: "Mail Schedule, Job Will Run In Next Minute" })

  }else{
      //Send To Only 1 Recipient
      sendMailForMarketing(readyData[0], attachments);
      return res.json({ status: true, message: "Message sent" })
  }

  
    
})

exports.marketingPushNativeNotification = asyncHandler(async (req, res, next) => {

  let template, recipients;

  if(req.body.templateId){
    template = await getMarketingTemplateById(req.body.templateId);
    if(!template) return res.json({ status: false, message: "Template not found or has been deleted"});

    if(req.body.recipients == "all-users" || req.body.recipients == "regular-users" || req.body.recipients == "agent-users" || req.body.recipients == "api-users"){
      recipients = await getUserFullDataByRecipientType(req.body.recipients);
    }
  }

  const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT));

  if(!recipients){
    // Parallel processing of recipients
    recipients = await getUsersByPushtokens(req.body.recipients);
  }
  

  const readyData = recipients.map(user => ({ token: user.pushToken, subject: processMessageTemplateForUser(template?.mt_subject || req.body.subject, user), message: processMessageTemplateForUser(template?.mt_content|| req.body.message, user) }))
    
  const notificationPromises = readyData.map(user => 
      limit(() => 
        pushNotification([user.token], {
          title: user.subject,
          body: user.message
        })
        .then(() => ({ success: true, token: user.token }))
        .catch(error => ({ 
          success: false, 
          token: user.token, 
          error: error.message 
        }))
      )
    );
  
    // 3. Execute all with progress tracking
    const results = await Promise.all(notificationPromises);
    
    const successCount = results.filter(result => result.success).length;
    const failureCount = results.filter(result => !result.success).length;

    // Provide a detailed response
    return res.json({
      status: successCount > 0,
      message: `Message sent. ${successCount} notification(s) delivered successfully, ${failureCount} failed.`,
      details: results
    });
    
})

/* PUSH SYSTEM NOTIFICATION */
exports.marketingPushSystemNotification = asyncHandler(async (req, res, next) => {

  let template, recipients;

  if(req.body.templateId){
    template = await getMarketingTemplateById(req.body.templateId);
    if(!template) return res.json({ status: false, message: "Template not found or has been deleted"});

    if(req.body.recipients == "all-users" || req.body.recipients == "regular-users" || req.body.recipients == "agent-users" || req.body.recipients == "api-users"){
      recipients = await getUserFullDataByRecipientType(req.body.recipients);
    }
  }

  const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT));

  if(!recipients){
    // Parallel processing of recipients
    recipients = await getUsersByIds(req.body.recipients);
  };
  
  const readyData = recipients.map(user => ({ ...user, subject: processMessageTemplateForUser(template?.mt_subject || req.body.subject, user), message: processMessageTemplateForUser(template?.mt_content || req.body.message, user) }));

  const notificationPromises = readyData.map(user => 
      limit(async () => {
        const notificationObj = {
          sn_user_id: user.uid,
          sn_type: "system",
          sn_title: user.subject,
          sn_message: user.message,
          sn_has_read: 0,
          sn_created_at: new Date()
        };
        
        const { insertId:systemNotificationId } = await createSystemNotification(notificationObj)
        notificationObj.sn_id = systemNotificationId;
        
        //Push Realtime System Notification To User
        ioSendSystemNotification(notificationObj);
        return { success: true }
      })
    );
  
    // 3. Execute all with progress tracking
    const results = await Promise.all(notificationPromises);
    
    const successCount = results.filter(result => result.success).length;
    const failureCount = results.filter(result => !result.success).length;

    // Provide a detailed response
    return res.json({
      status: successCount > 0,
      message: `Message sent. ${successCount} notification(s) delivered successfully, ${failureCount} failed.`,
      details: results
    });
    
})



